'use client';

import { useState, useEffect, useCallback, useRef, useMemo } from 'react';
import Link from 'next/link';
import Image from 'next/image';
import { usePathname } from 'next/navigation';
import {
  Layers,
  Menu,
  X,
  ChevronDown,
  Search,
  Bell,
  Linkedin,
  Youtube,
  Instagram,
  Sparkles,
  ArrowRight,
} from 'lucide-react';
import { useLanguage } from '@/contexts/language-context';
import { LanguageSwitcher } from './language-switcher';
import { Button } from './ui/button';
import {
  Sheet,
  SheetContent,
  SheetHeader,
  SheetTrigger,
} from '@/components/ui/sheet';
import {
  NavigationMenu,
  NavigationMenuContent,
  NavigationMenuItem,
  NavigationMenuLink,
  NavigationMenuList,
  NavigationMenuTrigger,
} from '@/components/ui/navigation-menu';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import { cn } from '@/lib/utils';
import { motion, AnimatePresence } from 'framer-motion';

// Types
interface NavItem {
  href: string;
  label: string;
  children?: NavItem[];
  badge?: string | number;
  icon?: React.ReactNode;
  description?: string;
}

interface HeaderProps {
  className?: string;
  showSearch?: boolean;
  showNotifications?: boolean;
}

const debounce = <F extends (...args: any[]) => any>(func: F, waitFor: number) => {
  let timeout: ReturnType<typeof setTimeout> | null = null;

  return (...args: Parameters<F>): Promise<ReturnType<F>> =>
    new Promise(resolve => {
      if (timeout) {
        clearTimeout(timeout);
      }

      timeout = setTimeout(() => resolve(func(...args)), waitFor);
    });
};

export function Header({
  className,
  showSearch = true,
  showNotifications = true
}: HeaderProps) {
  const { content, language } = useLanguage();
  const pathname = usePathname();
  const direction = language === 'ar' ? 'rtl' : 'ltr';

  // State
  const [isScrolled, setIsScrolled] = useState(false);
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);
  const [isSearchOpen, setIsSearchOpen] = useState(false);
  const [searchQuery, setSearchQuery] = useState('');
  const [hoveredNavItem, setHoveredNavItem] = useState<string | null>(null);
  const [isClient, setIsClient] = useState(false);

  useEffect(() => {
    setIsClient(true);
  }, []);

  // Refs
  const searchInputRef = useRef<HTMLInputElement>(null);
  const headerRef = useRef<HTMLDivElement>(null);

  // Navigation items with enhanced design
  const navItems: NavItem[] = useMemo(() => [
    {
      href: '/',
      label: content.nav.home || 'Home',
    },
    {
      href: '/about',
      label: content.nav.about,
      description: 'Learn about our consortium'
    },
    {
      href: '/services',
      label: content.nav.services,
    },
    {
      href: '/projects',
      label: content.nav.projects,
    },
    {
      href: '/capabilities',
      label: content.nav.capabilities || 'Consortium Capabilities'
    },
    {
      href: '/contact',
      label: content.nav.contact
    },
  ], [content, language]);

  // Debounced scroll handler
  useEffect(() => {
    const handleScroll = debounce(() => {
      const scrollY = window.scrollY;
      setIsScrolled(scrollY > 10);

      if (scrollY > 100 && isMobileMenuOpen) {
        setIsMobileMenuOpen(false);
      }
    }, 10);

    window.addEventListener('scroll', handleScroll, { passive: true });
    return () => window.removeEventListener('scroll', handleScroll);
  }, [isMobileMenuOpen]);

  // Close mobile menu on route change
  useEffect(() => {
    setIsMobileMenuOpen(false);
    setIsSearchOpen(false);
  }, [pathname]);

  // Keyboard shortcuts
  useEffect(() => {
    const handleKeyDown = (e: KeyboardEvent) => {
      if ((e.metaKey || e.ctrlKey) && e.key === 'k') {
        e.preventDefault();
        setIsSearchOpen(true);
        setTimeout(() => searchInputRef.current?.focus(), 100);
      }

      if (e.key === 'Escape') {
        setIsSearchOpen(false);
        setSearchQuery('');
      }
    };

    window.addEventListener('keydown', handleKeyDown);
    return () => window.removeEventListener('keydown', handleKeyDown);
  }, []);

  // Callbacks
  const handleSearchSubmit = useCallback((e: React.FormEvent) => {
    e.preventDefault();
    if (searchQuery.trim()) {
      console.log('Searching for:', searchQuery);
      setIsSearchOpen(false);
      setSearchQuery('');
    }
  }, [searchQuery]);

  const isActiveLink = useCallback((href: string) => {
    if (!isClient) return false;
    if (href === '/') return pathname === '/';
    if (href.includes('#')) {
      const [path, hash] = href.split('#');
      const currentPath = pathname === '/' ? '' : pathname;
      return currentPath === path && window.location.hash === `#${hash}`;
    }
    return pathname.startsWith(href);
  }, [pathname, isClient]);

  // Enhanced Logo Component with gradient animation
  const Logo = () => (
    <motion.div 
      initial={{ opacity: 0, x: -20 }} 
      animate={{ opacity: 1, x: 0 }} 
      transition={{ duration: 0.5 }}
      className="relative"
    >
      <Link
        href="/"
        className="group flex items-center gap-2 transition-all duration-300"
        aria-label="UBC NOOR Consortium Home"
      >
        <div className="relative h-20 w-48">
          <Image
            src="/ubc noor.png"
            alt="UBC NOOR Consortium Logo"
            fill
            className="object-contain"
            sizes="(max-width: 768px) 100vw, 33vw"
          />
        </div>
      </Link>
    </motion.div>
  );

  // Enhanced NavLink with hover effects
  const NavLink = ({ item, inSheet = false }: { item: NavItem; inSheet?: boolean }) => {
    const isActive = isActiveLink(item.href);
  
    if (item.children && !inSheet) {
      return (
        <NavigationMenu>
          <NavigationMenuList>
            <NavigationMenuItem>
              <NavigationMenuTrigger
                className={cn(
                  'font-semibold transition-all duration-300 bg-transparent text-sm',
                  'hover:text-accent',
                  isActive ? 'text-accent' : 'nav-text'
                )}
                onMouseEnter={() => setHoveredNavItem(item.href)}
                onMouseLeave={() => setHoveredNavItem(null)}
              >
                {item.label}
              </NavigationMenuTrigger>
              <NavigationMenuContent>
                <motion.ul
                  initial={{ opacity: 0, y: -10 }}
                  animate={{ opacity: 1, y: 0 }}
                  className="grid w-[400px] gap-2 p-4 bg-background shadow-2xl rounded-xl border"
                >
                  {item.children.map((child, index) => (
                    <motion.li
                      key={child.href}
                      initial={{ opacity: 0, x: -20 }}
                      animate={{ opacity: 1, x: 0 }}
                      transition={{ delay: index * 0.1 }}
                    >
                      <NavigationMenuLink asChild>
                        <Link
                          href={child.href}
                          passHref
                          className={cn(
                            'group block select-none rounded-lg p-4 leading-none no-underline outline-none transition-all duration-300',
                            'hover:bg-muted/50 hover:shadow-lg hover:scale-[1.02]',
                            isActiveLink(child.href) && 'bg-muted/50'
                          )}
                        >
                          <div className="flex items-center justify-between">
                            <div>
                              <div className="text-sm font-semibold mb-1 text-primary group-hover:text-accent transition-all">
                                {child.label}
                              </div>
                              {child.description && (
                                <p className="text-xs text-muted-foreground group-hover:text-foreground/80">
                                  {child.description}
                                </p>
                              )}
                            </div>
                            {child.icon && (
                              <motion.div whileHover={{ x: 5 }} className="text-accent">
                                {child.icon}
                              </motion.div>
                            )}
                          </div>
                        </Link>
                      </NavigationMenuLink>
                    </motion.li>
                  ))}
                </motion.ul>
              </NavigationMenuContent>
            </NavigationMenuItem>
          </NavigationMenuList>
        </NavigationMenu>
      );
    }
  
    return (
      <Link
        href={item.href}
        passHref
        aria-current={isActive ? 'page' : undefined}
        onClick={() => inSheet && setIsMobileMenuOpen(false)}
        onMouseEnter={() => setHoveredNavItem(item.href)}
        onMouseLeave={() => setHoveredNavItem(null)}
        className={cn(
          'relative font-semibold transition-all duration-300 group px-3 py-2 rounded-lg',
          inSheet ? 'block py-3 text-lg' : 'text-sm',
          isActive
            ? 'text-accent'
            : inSheet
            ? 'text-primary-foreground/80 hover:text-white hover:bg-white/10'
            : 'nav-text'
        )}
      >
        <span className="relative z-10 flex items-center gap-2">
          {item.icon}
          {item.label}
        </span>
        
        {!inSheet && (
          <motion.span 
            className="absolute bottom-0 left-0 right-0 h-0.5 bg-accent"
            initial={{ scaleX: 0 }}
            animate={{ scaleX: isActive ? 1 : 0 }}
            style={{ originX: 0.5 }}
            transition={{ duration: 0.3, ease: 'easeOut' }}
          />
        )}
      </Link>
    );
  };
  
  // Enhanced Search Bar
  const SearchBar = () => (
    <AnimatePresence>
      {isSearchOpen && (
        <motion.div
          initial={{ opacity: 0, scale: 0.95, y: -20 }}
          animate={{ opacity: 1, scale: 1, y: 0 }}
          exit={{ opacity: 0, scale: 0.95, y: -20 }}
          transition={{ duration: 0.3, type: "spring" }}
          className="absolute inset-x-4 top-1/2 -translate-y-1/2 z-50 md:inset-x-auto md:end-20 md:start-auto md:w-96"
        >
          <form onSubmit={handleSearchSubmit} className="relative">
            <Input
              ref={searchInputRef}
              type="search"
              placeholder={`${content.search || 'Search'}... (⌘K)`}
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="relative w-full pe-10 ps-10 bg-background/90 border-border/80 rounded-full shadow-lg"
              aria-label="Search"
            />
            <Search className="absolute start-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
            <Button
              type="button"
              variant="ghost"
              size="icon"
              className="absolute end-1 top-1/2 -translate-y-1/2 h-7 w-7 text-muted-foreground hover:bg-muted rounded-full"
              onClick={() => {
                setIsSearchOpen(false);
                setSearchQuery('');
              }}
            >
              <X className="h-4 w-4" />
            </Button>
          </form>
        </motion.div>
      )}
    </AnimatePresence>
  );

  // Enhanced Social Icons
  const SocialIcons = ({ inSheet = false }: { inSheet?: boolean }) => (
    <motion.div 
      initial={{ opacity: 0 }}
      animate={{ opacity: 1 }}
      transition={{ delay: 0.3 }}
      className={cn("flex items-center gap-1", inSheet && "mt-8 justify-center gap-3")}
    >
      {[
        { Icon: Linkedin, href: "#" },
        { Icon: Youtube, href: "#" },
        { Icon: Instagram, href: "#" }
      ].map(({ Icon, href }, index) => (
        <motion.div
          key={index}
          whileHover={{ scale: 1.1 }}
          whileTap={{ scale: 0.9 }}
        >
          <Link 
            href={href} 
            className={cn(
              "p-2 rounded-lg transition-all duration-300 social-icon hover:text-accent",
              inSheet ? 
                "text-white/80 hover:bg-white/10" :
                "text-muted-foreground hover:bg-muted"
            )}
          >
            <Icon className="h-5 w-5" />
          </Link>
        </motion.div>
      ))}
    </motion.div>
  );

  return (
    <>
      <motion.header
        ref={headerRef}
        initial={{ y: -100 }}
        animate={{ y: 0 }}
        transition={{ type: "spring", stiffness: 100 }}
        className={cn(
          'fixed top-0 z-50 w-full transition-all duration-300',
          isScrolled ? 'is-scrolled' : '',
          className
        )}
        role="banner"
      >
        <div className="container mx-auto relative">
          <div className="flex h-20 items-center justify-between px-4">
            <div className={cn(isSearchOpen ? 'opacity-0' : 'opacity-100 transition-opacity')}>
              <Logo />
            </div>

            {/* Desktop Navigation */}
            <motion.nav
              initial={{ opacity: 0 }}
              animate={{ opacity: 1 }}
              transition={{ duration: 0.5, delay: 0.2 }}
              className={cn(
                "hidden items-center gap-6 md:flex absolute left-1/2 -translate-x-1/2",
                 isSearchOpen ? 'opacity-0 pointer-events-none' : 'opacity-100'
              )}
              role="navigation"
              aria-label="Main navigation"
            >
              {navItems.map((item) => (
                  <NavLink key={item.href} item={item} />
              ))}
            </motion.nav>

            {/* Actions */}
            <div className={cn(
              "flex items-center gap-2",
              isSearchOpen ? 'opacity-0 pointer-events-none' : 'opacity-100'
            )}>
              {/* Search Button */}
              {showSearch && (
                <motion.div whileHover={{ scale: 1.05 }} whileTap={{ scale: 0.95 }}>
                  <Button
                    variant="ghost"
                    size="icon"
                    onClick={() => setIsSearchOpen(!isSearchOpen)}
                    aria-label="Toggle search"
                    className={cn(
                      "hidden md:flex rounded-lg transition-all duration-300 social-icon"
                    )}
                  >
                    <Search className="h-5 w-5" />
                  </Button>
                </motion.div>
              )}

              {/* Notifications */}
              {showNotifications && (
                <Button
                  variant="ghost"
                  size="icon"
                  className={cn(
                    "relative hidden md:flex rounded-lg transition-all duration-300 social-icon"
                  )}
                  aria-label="Notifications"
                >
                  <Bell className="h-5 w-5" />
                </Button>
              )}
              
              <div className={cn(
                "hidden md:flex items-center gap-2 text-muted-foreground"
              )}>
                 <div className="h-6 w-px bg-border"/>
                <LanguageSwitcher />
                <SocialIcons />
              </div>

              {/* Mobile Menu */}
              <div className="md:hidden">
                <Sheet open={isMobileMenuOpen} onOpenChange={setIsMobileMenuOpen}>
                  <SheetTrigger asChild>
                    <Button
                      variant="ghost"
                      size="icon"
                      aria-label="Open menu"
                       className={cn(
                        "rounded-lg transition-all duration-300 social-icon"
                       )}
                    >
                      <Menu className="h-6 w-6" />
                    </Button>
                  </SheetTrigger>
                  <SheetContent
                    side={direction === 'rtl' ? 'left' : 'right'}
                    className="w-[85vw] max-w-sm bg-primary text-primary-foreground border-0 flex flex-col"
                  >
                    <SheetHeader className="text-start border-b pb-4 border-white/20">
                       <Logo />
                    </SheetHeader>

                    {/* Mobile Search */}
                    {showSearch && (
                      <div className="mt-6 mb-4">
                        <Input
                          type="search"
                          placeholder={content.search || 'Search...'}
                          className="w-full bg-white/10 border-transparent placeholder:text-white/60 rounded-full text-white"
                        />
                      </div>
                    )}

                    {/* Mobile Navigation */}
                    <nav className="mt-8 flex flex-col gap-2" role="navigation">
                      {navItems.map((item) => (
                        <div key={item.href}>
                          <NavLink item={item} inSheet />
                          {item.children && (
                            <motion.div 
                              initial={{ height: 0, opacity: 0 }}
                              animate={{ height: "auto", opacity: 1 }}
                              className="ms-4 mt-2 space-y-2 border-s-2 border-white/20 ps-4"
                            >
                              {item.children.map((child) => (
                                <Link
                                  key={child.href}
                                  href={child.href}
                                  passHref
                                  onClick={() => setIsMobileMenuOpen(false)}
                                  className="block py-2 text-sm text-white/70 hover:text-accent transition-colors"
                                >
                                  {child.label}
                                </Link>
                              ))}
                            </motion.div>
                          )}
                        </div>
                      ))}
                    </nav>

                    <div className="mt-auto pt-8">
                      <div className="flex justify-center mb-4">
                        <LanguageSwitcher />
                      </div>
                      <SocialIcons inSheet />
                    </div>
                  </SheetContent>
                </Sheet>
              </div>
            </div>

            {/* Search Overlay */}
            <SearchBar />
          </div>
        </div>
      </motion.header>
    </>
  );
}
